<?php

namespace ZPOS\Admin\Stations\Tabs;

use ZPOS\Admin\Setting\Box;
use ZPOS\Admin\Setting\Input\Checkbox;
use ZPOS\Admin\Setting\Input\Description;
use ZPOS\Admin\Setting\Input\Select;
use ZPOS\Admin\Setting\Input\TaxArray;
use ZPOS\Admin\Setting\PostTab;
use ZPOS\Admin\Stations\Setting;
use ZPOS\Station;

class Tax extends PostTab
{
	public $name = 'Tax';
	public $path = '/tax';

	public function __construct()
	{
		parent::__construct();
		$this->name = __('Tax', 'point-of-sale-pos-woocommerce');
		add_filter(PostTab::class . '::getValueByPost', [$this, 'getValueForWCStation'], 10, 3);
	}

	public function getBoxes()
	{
		return [
			new Box(
				__('Tax Options', 'point-of-sale-pos-woocommerce'),
				['ignore' => !Setting::isWCStationEdit()],
				new Description(
					sprintf(
						'%s <a href="%s">%s</a>',
						__('Default WooCommerce Station use', 'point-of-sale-pos-woocommerce'),
						add_query_arg(['page' => 'wc-settings', 'tab' => 'tax'], admin_url('admin.php')),
						__('WooCommerce Settings', 'point-of-sale-pos-woocommerce')
					),
					__('Tax Calculation', 'point-of-sale-pos-woocommerce')
				)
			),
			new Box(
				__('Tax Options', 'point-of-sale-pos-woocommerce'),
				['ignore' => Setting::isWCStationEdit()],
				new Select(
					__('Tax Options', 'point-of-sale-pos-woocommerce'),
					'pos_tax_enabled',
					$this->getValue('pos_tax_enabled'),
					self::get_enabled_values()
				)
			),
			new Box(
				null,
				['withoutBreak' => true],
				new Checkbox(
					null,
					'pos_tax_vat_number',
					$this->getValue('pos_tax_vat_number'),
					__('Enable Tax/VAT Number on Order', 'point-of-sale-pos-woocommerce')
				)
			),
			new Box(
				null,
				['withoutBreak' => true],
				new Select(
					__('Display prices in the POS', 'point-of-sale-pos-woocommerce'),
					'pos_display_prices_include_tax_in_shop',
					$this->getValue('pos_display_prices_include_tax_in_shop'),
					self::get_display_price_values()
				),
				new Select(
					__('Display prices during POS cart', 'point-of-sale-pos-woocommerce'),
					'pos_display_prices_include_tax_in_cart',
					$this->getValue('pos_display_prices_include_tax_in_cart'),
					self::get_display_price_values()
				)
			),
			new Box(
				null,
				[
					'ignore' => Setting::isWCStationEdit(),
					'withoutBreak' => true,
				],
				new TaxArray(
					__('Calculate Tax Based on', 'point-of-sale-pos-woocommerce'),
					'pos_tax_based_on_order',
					$this->get_tax_based_on(),
					[
						'sanitize' => [$this, 'sanitizeTax'],
					]
				)
			),
			new Box(
				null,
				['ignore' => !Setting::isWCStationEdit(), 'withoutBreak' => true],
				new Description(
					sprintf(
						'%s <a href="%s">%s</a>',
						__('Default WooCommerce Station use', 'point-of-sale-pos-woocommerce'),
						add_query_arg(['page' => 'wc-settings', 'tab' => 'tax'], admin_url('admin.php')),
						__('WooCommerce Settings', 'point-of-sale-pos-woocommerce')
					),
					'Calculate Tax Based on'
				)
			),
		];
	}

	protected function get_tax_based_on()
	{
		return function ($post) {
			$values = self::get_calculate_on_values();
			return array_map(function ($value) use ($values) {
				return ['value' => $value, 'label' => $values[$value]];
			}, $this->getValue('pos_tax_based_on_order', $post));
		};
	}

	public static function getDefaultValue($value, $post, $name)
	{
		switch ($name) {
			case 'pos_tax_enabled':
				return 'on';
			case 'pos_display_prices_include_tax_in_shop':
			case 'pos_display_prices_include_tax_in_cart':
				return 'yes';
			case 'pos_tax_based_on_order':
				return ['shipping', 'billing', 'pos', 'wc'];
			default:
				return $value;
		}
	}

	public function getValueForWCStation($value, $post, $name)
	{
		if (Station::isWCStation($post->ID)) {
			switch ($name) {
				case 'pos_tax_enabled':
					return get_option('woocommerce_calc_taxes') ? 'on' : 'off';
				case 'pos_tax_based_on_order':
					$tax_based_on = get_option('woocommerce_tax_based_on');
					if ($tax_based_on === 'base') {
						return ['wc'];
					} else {
						return [$tax_based_on, 'wc'];
					}
				default:
					return $value;
			}
		}
		return $value;
	}

	public static function sanitizeTax($raw_data)
	{
		$keys = array_keys(self::get_calculate_on_values());
		$data = array_filter($raw_data, function ($element) use ($keys) {
			return in_array($element, $keys);
		});
		return $data;
	}

	public static function get_enabled_values()
	{
		return [
			[
				'value' => 'on',
				'label' => __(
					'Enabled (using WooCommerce configurations)',
					'point-of-sale-pos-woocommerce'
				),
			],
			[
				'value' => 'off',
				'label' => __('Disabled', 'point-of-sale-pos-woocommerce'),
			],
		];
	}

	public function get_display_price_values()
	{
		return [
			['value' => 'yes', 'label' => __('Including tax', 'point-of-sale-pos-woocommerce')],
			['value' => 'no', 'label' => __('Excluding tax', 'point-of-sale-pos-woocommerce')],
		];
	}

	public static function get_calculate_on_values()
	{
		return [
			'pos' => __('POS base address', 'point-of-sale-pos-woocommerce'),
			'wc' => __('Default WooCommerce Shop Base', 'point-of-sale-pos-woocommerce'),
			'shipping' => __('Customer shipping address', 'point-of-sale-pos-woocommerce'),
			'billing' => __('Customer billing address', 'point-of-sale-pos-woocommerce'),
		];
	}
}
